﻿#region << Using Directives >>
using System;
using Volpe.Cafe.Data;
#endregion

namespace Volpe.Cafe.Generic
{
    /// <summary>
    /// Represents an object that stores compliance modeling data based on a combination of vehicle's class and style.
    /// </summary>
    [Serializable]
    public class VCXValue<T>: ICloneable
    {

        #region /*** Ctors ***/

        /// <summary>
        /// Initializes a new instance of the <see cref="VCXValue{T}"/> class.
        /// </summary>
        public VCXValue()
        {
            this.Items = new T[Names.Length];
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="VCXValue{T}"/> class using the specified values.
        /// </summary>
        public VCXValue(T cars, T vans, T suvs, T pickups, T class12a, T zevs, T class2b3)
        {
            this.Items = new T[] { cars, vans, suvs, pickups, class12a, zevs, class2b3 };
        }

        #endregion

        #region /*** Methods ***/

        #region /* Overloaded Operators */

        /// <summary>
        /// Determines whether the specified <see cref="VCXValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="VCXValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool operator ==(VCXValue<T> value1, VCXValue<T> value2)
        {
            return Equals(value1, value2);
        }
        /// <summary>
        /// Determines whether the specified <see cref="VCXValue{T}"/> values are not equal.
        /// </summary>
        /// <param name="value1">The first value to compare.</param>
        /// <param name="value2">The second value to compare.</param>
        /// <returns>true, if the two <see cref="VCXValue{T}"/> values are not equal; false, otherwise.</returns>
        public static bool operator !=(VCXValue<T> value1, VCXValue<T> value2)
        {
            return !Equals(value1, value2);
        }

        #endregion

        #region /* ICloneable Members */

        /// <summary>
        /// Creates a shallow copy of the current <see cref="VCXValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="VCXValue{T}"/>.</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }
        /// <summary>
        /// Creates a shallow copy of the current <see cref="VCXValue{T}"/> instance.
        /// </summary>
        /// <returns>A new object that is a copy of this <see cref="VCXValue{T}"/>.</returns>
        public VCXValue<T> Clone()
        {
            VCXValue<T> value = new VCXValue<T>();
            this.CopyTo(value);
            return value;
        }
        /// <summary>
        /// Copies the members of the current <see cref="VCXValue{T}"/> instance into the specified value.
        /// </summary>
        /// <param name="value">The value where to copy the members of the current instance.</param>
        protected void CopyTo(VCXValue<T> value)
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                value.Items[i] = this.Items[i];
            }
        }

        #endregion

        #region /* Overriden from object */

        /// <summary>
        /// Returns the string representation of this <see cref="VCXValue{T}"/> instance.
        /// </summary>
        /// <returns>The string representation of the <see cref="VCXValue{T}"/> instance.</returns>
        public override string ToString()
        {
            string s = string.Empty;
            for (int i = 0; i < this.Items.Length; i++)
            {
                if (i > 0) { s += ", "; }
                s += (Names[i] + "=" + this.Items[i].ToString());
            }
            return "{" + s + "}";
        }

        /// <summary>
        /// Serves as a hash function for a particular type, suitable for use in hashing algorithms and data structures like a hash table.
        /// </summary>
        /// <returns>A hash code for the current <see cref="VCXValue{T}"/>.</returns>
        public override int GetHashCode()
        {
            int hash = 0;
            for (int i = 0; i < this.Items.Length; i++)
            {
                hash = hash ^ this.Items[i].GetHashCode();
            }
            return hash;
        }

        /// <summary>
        /// Determines whether the specified <see cref="Object"/> is equal to the current <see cref="VCXValue{T}"/> value.
        /// </summary>
        /// <param name="obj">The <see cref="Object"/> to compare with the current <see cref="VCXValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="Object"/> is equal to the current <see cref="VCXValue{T}"/> value; false, otherwise.</returns>
        public override bool Equals(object obj)
        {
            return (obj is VCXValue<T>) ? this.Equals((VCXValue<T>)obj) : base.Equals(obj);
        }
        /// <summary>
        /// Determines whether the specified <see cref="VCXValue{T}"/> value is equal to the current <see cref="VCXValue{T}"/> value.
        /// </summary>
        /// <param name="value">The <see cref="VCXValue{T}"/> value to compare with the current <see cref="VCXValue{T}"/> value.</param>
        /// <returns>true, if the specified <see cref="VCXValue{T}"/> value is equal to the current <see cref="VCXValue{T}"/> value;
        ///   false, otherwise.</returns>
        public bool Equals(VCXValue<T> value)
        {
            return Equals(this, value);
        }
        /// <summary>
        /// Determines whether the specified <see cref="VCXValue{T}"/> values are equal.
        /// </summary>
        /// <param name="value1">The first <see cref="VCXValue{T}"/> value to compare.</param>
        /// <param name="value2">The second <see cref="VCXValue{T}"/> value to compare.</param>
        /// <returns>true, if the specified <see cref="VCXValue{T}"/> values are equal; false, otherwise.</returns>
        public static bool Equals(VCXValue<T> value1, VCXValue<T> value2)
        {
            if (object.ReferenceEquals(value1, null) && object.ReferenceEquals(value2, null)) { return true; }
            if (object.ReferenceEquals(value1, null) || object.ReferenceEquals(value2, null)) { return false; }
            //
            for (int i = 0; i < value1.Items.Length; i++)
            {
                if (!value1.Items[i].Equals(value2.Items[i])) { return false; }
            }
            return true;
        }

        #endregion

        /// <summary>
        /// Resets all members of this <see cref="VCXValue{T}"/> instance to their default values (0, false, or null).
        /// </summary>
        public void Clear()
        {
            for (int i = 0; i < this.Items.Length; i++)
            {
                this.Items[i] = default(T);
            }
        }

        /// <summary>
        /// Returns the index corresponding to the specified <see cref="VehicleClass"/> and <see cref="VehicleStyle"/>.
        /// </summary>
        /// <returns>The index corresponding to the specified <see cref="VehicleClass"/> and <see cref="VehicleStyle"/>.</returns>
        protected int GetIndex(VehicleClass vehClass, VehicleStyle vehStyle, HEVType hevType)
        {
            bool isZev = (hevType == HEVType.PureElectric || hevType == HEVType.FuelCell);
            //
            switch (vehClass)
            {
                case VehicleClass.LDV   : return (isZev) ? 5 : 0;
                //
                case VehicleClass.LDT1  :
                case VehicleClass.LDT2a :
                case VehicleClass.LDT12a: return (isZev                                ) ? 5 :
                                                 (vehStyle == VehicleStyle.Minivan      ||
                                                  vehStyle == VehicleStyle.PassengerVan ||
                                                  vehStyle == VehicleStyle.CargoVan     ||
                                                  vehStyle == VehicleStyle.Van         ) ? 1 :
                                                 (vehStyle == VehicleStyle.SportUtility) ? 2 :
                                                 (vehStyle == VehicleStyle.Pickup      ) ? 3 : 4;
                //
                case VehicleClass.LDT2b :
                case VehicleClass.LDT3  :
                case VehicleClass.LDT2b3: return 6;
                //
                default: throw new ArgumentOutOfRangeException("vehClass", "Vehicle class is not specified or not supported.");
            }
        }

        #endregion

        #region /*** Properties ***/

        /// <summary>Gets or sets the compliance modeling data value based on the class and style of the given vehicle.</summary>
        /// <exception cref="System.ArgumentOutOfRangeException">The value specified does not represent a valid vehicle class and style combination.</exception>
        public T this[Vehicle veh]
        {
            get { return this.Items[this.GetIndex(veh.VehicleClass, veh.VehicleStyle, veh.HEVType)]; }
            set { this.Items[this.GetIndex(veh.VehicleClass, veh.VehicleStyle, veh.HEVType)] = value; }
        }
        /// <summary>Gets or sets the compliance modeling data value based on the given vehicle class and style.</summary>
        /// <exception cref="System.ArgumentOutOfRangeException">The value specified does not represent a valid vehicle class and style combination.</exception>
        public T this[VehicleClass vehClass, VehicleStyle vehStyle, HEVType hevType]
        {
            get { return this.Items[this.GetIndex(vehClass, vehStyle, hevType)]; }
            set { this.Items[this.GetIndex(vehClass, vehStyle, hevType)] = value; }
        }

        /// <summary>Gets an array of compliance modeling data specified by <see cref="VehicleClass"/> and <see cref="VehicleStyle"/>.</summary>
        public T[] Items { get; private set; }

        #endregion

        #region /*** Variables ***/

        /// <summary>Provides the friendly name for each of the supported vehicle class/style combinations. This field is read-only.</summary>
        public static readonly string[] Names = new string[] { "Cars",
                                                               "Vans",
                                                               "SUVs",
                                                               "Pickups",
                                                               "Class 1/2a Average",
                                                               "ZEVs",
                                                               "Class 2/b3" };

        #endregion

    }
}
